from enum import Enum
import os
import logging
import asyncio
import typing
from ccxt.base import errors as ccxt_errors
import ccxt.pro as ccxt
from contextlib import suppress
from ccxt import DECIMAL_PLACES

from decimal import Decimal, ROUND_DOWN
from datetime import datetime, timedelta, timezone
from tortoise import connections
import json
from .exceptions import ECUnexpectedError, ECInvalidValue

log = logging.getLogger("farmer")  # Use the same logger name as in other parts of the project

"""
TODO: improve this client so that we cache user specific clients for a certain amount of time. Make it so that 
in order to create() client we just need to pass uid then within create() func we fetch the keys from the db,
create the client and cache it for a certain amount of time.

Do no forget to also close the client after cache expires!
"""


class ExchangeEnum(str, Enum):
    BYBIT = "BYBIT"
    WOO = "WOO"
    HYPERLIQUID = "HYPERLIQUID"
    BINANCE = "BINANCE"
    KRAKEN = "KRAKEN"


class MarketTypeEnum(str, Enum):
    SPOT = "spot"
    FUTURES = "future"
    SWAP = "swap"
    OPTIONS = "option"


class SideEnum(str, Enum):
    BUY = "buy"
    SELL = "sell"


class ZCCXT:
    client = (
        None  # unable to set ccxt.Exchange type because all methods in that base class raise NotImplemented exception
    )

    def __init__(
        self,
        api_key: str,
        api_secret: str,
        exchange: ExchangeEnum,
        is_demo=False,
        identifier=None,
        disable_db=False,
    ):
        if exchange.value not in [e.value for e in ExchangeEnum]:
            raise ECInvalidValue(f"Invalid exchange provided: {exchange}")

        self.api_key = api_key
        self.api_secret = api_secret
        self.exchange = exchange
        self.is_demo = is_demo
        self.identifier = identifier
        self.db_disabled = disable_db

    @classmethod
    async def create(
        cls,
        api_key: str,
        api_secret: str,
        exchange: ExchangeEnum,
        is_demo=False,
        identifier=None,
        disable_db=False,
    ):
        self = cls(api_key, api_secret, exchange, is_demo, identifier, disable_db)

        params = {
            "apiKey": self.api_key,
            "secret": self.api_secret,
            # retry for 1 minutes (20 * 3sec = 60sec)
            "maxRetriesOnFailure": 20,
            "maxRetriesOnFailureDelay": 3000,
        }

        if exchange == ExchangeEnum.HYPERLIQUID:
            params.pop("secret", None)
            params.update(
                {
                    "walletAddress": self.identifier,
                    "enableRateLimit": True,
                    # "rateLimit": 1200,
                    "privateKey": self.api_secret,
                }
            )
        elif exchange == ExchangeEnum.WOO:
            params.update({"uid": self.identifier})

        self.client = getattr(ccxt, self.exchange.lower())(params)

        socks_proxy = os.environ.get("SOCKS_PROXY", None)
        if socks_proxy:
            self.client.socks_proxy = socks_proxy
            self.client.ws_socks_proxy = socks_proxy

        if not self.db_disabled:
            conn = connections.get("default")
            _, res = await conn.execute_query(
                "SELECT * FROM ccxt_market_cache WHERE exchange = $1",
                (self.exchange.value,),
            )
        else:
            res = []

        if len(res) > 0:
            key = res[0]
            if key["updated_at"] <= datetime.now(timezone.utc) - timedelta(hours=3):
                markets = await self.client.load_markets(reload=True)
                await self.cache_data(markets)
            else:
                self.client.set_markets(json.loads(key["data"]))
        else:
            markets = await self.client.load_markets(reload=True)
            await self.cache_data(markets)

        if self.is_demo:
            self.client.enableDemoTrading(True)

        return self

    async def cache_data(self, data):
        if self.db_disabled:
            return
        conn = connections.get("default")
        await conn.execute_query(
            """
            INSERT INTO ccxt_market_cache(exchange, data) VALUES ($1, $2)
            ON CONFLICT (exchange) DO UPDATE SET data = $2
            """,
            (self.exchange, json.dumps(data)),
        )

    async def close(self):
        await self.client.close()

    async def get_balance(self):
        """
        Fetch balances from the exchange
        """
        tasks = [self.client.fetch_balance()]

        if self.exchange == ExchangeEnum.BYBIT:
            tasks.append(self.client.privateGetV5AccountInfo())
            tasks.append(self.get_positions())
        elif self.exchange == ExchangeEnum.WOO:
            tasks.append(self.client.v3PrivateGetAccountinfo())
            tasks.append(self.get_positions())
        elif self.exchange == ExchangeEnum.HYPERLIQUID:
            # for HL we need to fetch balance twice, once for futures and another time for spot
            tasks.append(self.client.fetch_balance(params={"type": "spot"}))
            tasks.append(self.get_positions())
        elif self.exchange == ExchangeEnum.BINANCE:
            tasks.append(self.client.fetch_balance(params={"type": "future"}))
            tasks.append(self.get_positions())

        responses = await asyncio.gather(*tasks)
        response_data = {
            "balance": responses[0],
        }

        if self.exchange == ExchangeEnum.BYBIT:
            response_data["account_data"] = responses[1]["result"]
            response_data["positions"] = responses[2]
        elif self.exchange == ExchangeEnum.WOO:
            response_data["account_data"] = responses[1]["data"]
            response_data["positions"] = responses[2]
        elif self.exchange == ExchangeEnum.HYPERLIQUID:
            response_data["account_data"] = {}  # NOTE: not available on HL
            response_data["balance_spot"] = responses[1]
            response_data["positions"] = responses[2]
        elif self.exchange == ExchangeEnum.BINANCE:
            response_data["account_data"] = {}
            response_data["balance_future"] = responses[1]
            response_data["positions"] = responses[2]

        return self.handle_balance_response(response_data)

    async def get_positions(self, params=None):
        """
        Fetch positions from the exchange
        """
        params = params if params else {}
        return await self.client.fetch_positions(params=params)

    async def wait_for_order_fill(self, order_id: str, market: str):
        """
        Wait for an order to be filled
        """
        resp = None
        max_retries = 5

        while True:
            if max_retries == 0:
                if resp is not None:
                    raise ECUnexpectedError(
                        f"Exited after 5 retries. Order {order_id} not in closed state. Current state is {resp['state']}."
                    )
                raise ECUnexpectedError(f"Exited after 5 retries. Unable to get order {order_id} from api.")

            try:
                resp = await self.get_order(order_id, market)
            except ccxt_errors.OrderNotFound:
                log.debug("Order %s not found, retries left: %d", order_id, max_retries)
                max_retries -= 1
                await asyncio.sleep(0.5)
                continue

            if resp["status"] == "closed":
                # order filled
                return resp
            if resp["status"] == "cancelled":
                log.error("Order %s cancelled. Filled %s/%s", resp["id"], resp["filled"], resp["amount"])
                return resp

            log.debug(
                "Order %s not filled yet; %s/%s; status: %s. Checking again soon.",
                order_id,
                resp["filled"],
                resp["amount"],
                resp["status"],
            )
            await asyncio.sleep(0.1)

    async def get_orderbook(self, symbol: str):
        """
        Fetch orderbook from the exchange
        """
        return await self.client.fetch_order_book(symbol)

    async def _get_order(
        self,
        order_id: str,
        market: typing.Optional[MarketTypeEnum] = MarketTypeEnum.SPOT,
        order_type: typing.Optional[typing.Union["open", "closed"]] = None,
        symbol: typing.Optional[str] = None,
    ):
        try:
            if self.exchange == ExchangeEnum.BYBIT:
                market_value = market.value
                if market_value == "future" or market_value == "swap":
                    market_value = "linear"

                if order_type == "open":
                    return await self.client.fetch_open_order(order_id, params={"category": market_value})
                elif order_type == "close":
                    return await self.client.fetch_closed_order(order_id, params={"category": market_value})
                else:
                    # we want to suppress the 1st exception, but we're ok to raise an error for the 2nd call
                    # so it gets captured one level above
                    with suppress(ccxt_errors.OrderNotFound):
                        return await self.client.fetch_open_order(order_id, params={"category": market_value})
                    return await self.client.fetch_closed_order(order_id, params={"category": market_value})
            elif self.exchange == ExchangeEnum.BINANCE:
                return await self.client.fetch_order(order_id, symbol)
            else:
                return await self.client.fetch_order(order_id)
        except ccxt_errors.OrderNotFound:
            return None
        except ccxt_errors.BadRequest as e:
            assert True
            if "order can not be found" in e.args[0]:
                return None
            raise e
        except Exception as e:
            raise e

    async def get_order(
        self,
        order_id: str,
        market: typing.Optional[MarketTypeEnum] = MarketTypeEnum.SPOT,
        order_type: typing.Optional[typing.Union["open", "closed"]] = None,
        symbol: typing.Optional[str] = None,
    ):
        """
        Fetch an order from the exchange
        """
        if self.exchange == ExchangeEnum.BYBIT:
            return await self._get_order(order_id, market, order_type)
        elif self.exchange == ExchangeEnum.WOO:
            return await self._get_order(order_id)
        elif self.exchange == ExchangeEnum.HYPERLIQUID:
            order = await self._get_order(order_id)
            if not order or not order["id"]:
                return None

            # set default values in case we return early
            order.update(
                {
                    "average_price": 0,
                    "fee": {},
                    "fees": [],
                }
            )

            # NOTE: sux that we need this, but HL doesnt return fee info directly when fetching an order ...
            if order["filled"] > 0:
                trades = await self.get_order_trades(order_id, market)
                # its happenign way too often that an order does not have any trades associated with it, seems like
                # HL takes time to return trades for a given order ..
                if len(trades) == 0:
                    log.warning("Didnt fetch any trades for HL order id %s", order_id)
                    return order

                average_price = sum(trade["price"] for trade in trades) / len(trades)
                total_fee_cost = sum(trade["fee"]["cost"] for trade in trades)

                # Sum of costs in 'fees', grouped by currency
                fees_by_currency_dict = {}
                for trade in trades:
                    for fee in trade["fees"]:
                        currency = fee["currency"]
                        if currency not in fees_by_currency_dict:
                            fees_by_currency_dict[currency] = {"currency": currency, "cost": 0}
                        fees_by_currency_dict[currency]["cost"] += fee["cost"]

                fees_by_currency = list(fees_by_currency_dict.values())

                order["average"] = average_price
                order["fee"] = {
                    "currency": trades[0]["fee"]["currency"],
                    "cost": total_fee_cost,
                }
                order["fees"] = fees_by_currency

            return order
        elif self.exchange == ExchangeEnum.BINANCE:
            order = await self._get_order(order_id, symbol=symbol)
            # set default values in case we return early
            order.update(
                {
                    "average_price": 0,
                    "fee": {},
                    "fees": [],
                }
            )
            # NOTE: sux that we need this, but Binance doesnt return fee info directly when fetching an order ...
            if order["filled"] > 0:
                trades = await self.get_order_trades(order_id, market, symbol=symbol)
                if len(trades) == 0:
                    log.warning("Didnt fetch any trades for HL order id %s", order_id)
                    return order

                average_price = sum(trade["price"] for trade in trades) / len(trades)
                total_fee_cost = sum(trade["fee"]["cost"] for trade in trades)

                # Sum of costs in 'fees', grouped by currency
                fees_by_currency_dict = {}
                for trade in trades:
                    for fee in trade["fees"]:
                        currency = fee["currency"]
                        if currency not in fees_by_currency_dict:
                            fees_by_currency_dict[currency] = {"currency": currency, "cost": 0}
                        fees_by_currency_dict[currency]["cost"] += fee["cost"]

                fees_by_currency = list(fees_by_currency_dict.values())

                order["average"] = average_price
                order["fee"] = {
                    "currency": trades[0]["fee"]["currency"],
                    "cost": total_fee_cost,
                }
                order["fees"] = fees_by_currency

            return order
        else:
            raise ECInvalidValue(f"Unknown exchange {self.exchange} provided")

    async def get_order_trades(
        self, order_id, market: typing.Optional[MarketTypeEnum] = MarketTypeEnum.SPOT, symbol=None
    ):
        """
        Fetch trades for an order
        """
        if self.exchange == ExchangeEnum.BYBIT:
            market_value = market.value
            if market_value == "future" or market_value == "swap":
                market_value = "linear"
            return await self.client.fetch_order_trades(order_id, params={"category": market_value})
        elif self.exchange == ExchangeEnum.WOO:
            order = await self.get_order(order_id, market)
            return await self.client.fetch_order_trades(order_id, order["symbol"])
        elif self.exchange == ExchangeEnum.HYPERLIQUID:
            order = await self.client.fetch_order(order_id)
            # timestamp attr returned directly by CCXT does uses a different value: statusTimestamp instead of timestamp
            timestamp = int(order["info"]["order"]["timestamp"])

            # TODO: no pagination implemented, HL max returned items is 500 which shouldn't really be a problem here ...
            trades = await self.client.fetch_my_trades(order["symbol"], since=timestamp)

            filtered_trades = []
            for trade in trades:
                if trade["order"] != order_id:
                    continue

                # due to HL not providing this data we have to trust ourselves that orders indeed are of type "maker"
                if not trade.get("takerOrMaker"):
                    trade.update({"takerOrMaker": "maker"})
                filtered_trades.append(trade)
            return filtered_trades
        elif self.exchange == ExchangeEnum.BINANCE:
            order = await self.client.fetch_order(order_id, symbol)
            # timestamp attr returned directly by CCXT does uses a different value: statusTimestamp instead of timestamp
            timestamp = int(order["timestamp"])

            trades = await self.client.fetch_my_trades(order["symbol"], since=timestamp)
            filtered_trades = []
            for trade in trades:
                if trade["order"] != order_id:
                    continue

                filtered_trades.append(trade)
            return filtered_trades
        else:
            raise ECInvalidValue(f"Unknown exchange {self.exchange} provided")

    async def create_order(self, symbol, side: SideEnum, amount: Decimal, reduce_only: typing.Optional[bool] = None):
        """
        Create an order on the exchange
        """
        # todo: contract size pitfall ...
        params = {}
        if reduce_only:
            if self.exchange == ExchangeEnum.BYBIT:
                params.update({"isReduceOnly": reduce_only})
            elif self.exchange == ExchangeEnum.WOO:
                params.update({"reduceOnly": reduce_only})

        params.update({"createMarketBuyOrderRequiresPrice": False, "createMarketSellOrderRequiresPrice": False})
        data = await self.client.create_order(symbol, type="market", side=side.value, amount=amount, params=params)
        return data

    async def create_limit_order(
        self, symbol, side: SideEnum, amount: Decimal, price: Decimal, reduce_only: typing.Optional[bool] = None
    ):
        """
        Create a limit order (we want to be a maker not a taker to cash in on lower fees) on the exchange
        """
        # todo: contract size pitfall ...
        params = {"timeInForce": "PO", "postOnly": True}
        if reduce_only:
            params.update({"reduceOnly": reduce_only})

        if self.exchange == ExchangeEnum.HYPERLIQUID:
            del params["timeInForce"]

        try:
            live_order = await self.client.create_limit_order(
                symbol, side=side.value, amount=amount, price=price, params=params
            )
        except ccxt_errors.InvalidOrder as e:
            if self.exchange == ExchangeEnum.HYPERLIQUID:
                if "Post only order would have immediately matched" in e.args[0]:
                    log.debug(e.args[0])
                    return None
            raise e

        # Start: because HL doesnt return these values when creating order ...
        if not live_order.get("price"):
            live_order["price"] = price
        if not live_order.get("amount"):
            live_order["amount"] = amount
        if not live_order.get("side"):
            live_order["side"] = side.value
        # End
        return live_order

    async def create_market_order(self, symbol, side, amount, reduce_only: typing.Optional[bool] = None):
        params = {}
        if reduce_only:
            params.update({"reduceOnly": reduce_only})

        # price is required for some exchanges in order to calc. slippage
        slippage = self.client.options.get("defaultSlippage", 0)
        orderbook = await self.get_orderbook(symbol)
        if side == SideEnum.SELL:
            base_price = orderbook["bids"][0][0]
            # For sells, reduce price by slippage percentage
            price = base_price * (1 - slippage)
        else:
            base_price = orderbook["asks"][0][0]
            # For buys, increase price by slippage percentage
            price = base_price * (1 + slippage)

        # Convert to Decimal and round to appropriate precision
        price = Decimal(str(price)).quantize(Decimal("0.00000001"))

        log.info("Creating %s order for %s with price=%s amount=%s", side, symbol, price, amount)
        try:
            live_order = await self.client.create_order(
                symbol, "market", side=side.value, amount=amount, price=price, params=params
            )
        except ccxt_errors.InvalidOrder as e:
            if self.exchange == ExchangeEnum.HYPERLIQUID:
                if "Post only order would have immediately matched" in e.args[0]:
                    return None
            raise e

        # Start: because HL doesnt return these values when creating order ...
        if not live_order.get("price"):
            live_order["price"] = price
        if not live_order.get("amount"):
            live_order["amount"] = amount
        if not live_order.get("side"):
            live_order["side"] = side.value
        # End

        return live_order

    async def edit_order(
        self,
        order_id,
        symbol,
        side: SideEnum,
        amount: Decimal,
        price: Decimal,
        reduce_only: typing.Optional[bool] = None,
    ):
        """
        Edit an order on the exchange
        """
        # todo: contract size pitfall ...
        params = {"timeInForce": "PO", "postOnly": True}
        if reduce_only:
            params.update({"reduceOnly": reduce_only})

        if self.exchange == ExchangeEnum.HYPERLIQUID:
            del params["timeInForce"]

        try:
            data = await self.client.edit_order(
                order_id, symbol, type="limit", side=side, amount=amount, price=price, params=params
            )
        except ccxt_errors.InvalidOrder as e:
            if self.exchange == ExchangeEnum.HYPERLIQUID:
                if "Post only order would have immediately matched" in e.args[0]:
                    return None
            raise e
        return data

    async def create_spot_market_order_with_cost(self, symbol: str, side: SideEnum, cost: Decimal, params: any = None):
        """
        Create a spot market order with a specific cost
        """
        if side == SideEnum.BUY:
            return await self.client.create_market_buy_order_with_cost(symbol, cost, params=params or {})

        return await self.client.create_market_sell_order_with_cost(symbol, cost, params=params or {})

    async def get_leverage(self, symbol: str):
        """
        Fetch leverage for a symbol from the exchange
        """
        params = {}

        if self.exchange == ExchangeEnum.HYPERLIQUID:
            # HL does not have an endpoint for this yet, so we have them hardcoded to one of the lowest amounts for
            # safety reasons
            return {"long": Decimal(3), "short": Decimal(3)}

        if self.exchange == ExchangeEnum.WOO:
            params.update({"position_mode": "HEDGE_MODE"})
        elif self.exchange == ExchangeEnum.BYBIT:
            params.update({"category": "linear"})

        data = await self.client.fetch_leverage(symbol, params=params)
        return {
            "long": Decimal(data["longLeverage"]),
            "short": Decimal(data["shortLeverage"]),
        }

    async def get_funding_fees_history(
        self,
        symbol: typing.Optional[str] = None,
        from_dt: typing.Optional[datetime] = None,
        page_size=25,
        next_page=None,
    ):
        items = []
        if self.exchange == ExchangeEnum.BYBIT:
            params = {"category": "linear", "execType": "Funding"}
            if symbol:
                params["symbol"] = symbol
            if from_dt:
                params["startTime"] = int(from_dt.timestamp() * 1000)
            if page_size:
                params["limit"] = page_size
            if next_page:
                params["cursor"] = next_page
            data = await self.client.privateGetV5ExecutionList(params)
            results = data["result"]["list"]
            for item in results:
                # convert execTime (unix timestamp) to datetime
                created_at = datetime.utcfromtimestamp(float(item["execTime"]) / 1000)

                if item["execType"] != "Funding":
                    continue

                # get our unified ticker name
                ticker_name = None
                markets = self.client.markets_by_id[item["symbol"]]
                for market in markets:
                    if market["type"] == "swap":
                        ticker_name = market["symbol"]
                        break

                if not ticker_name:
                    log.warning("No market found for %s on exchange %s", item["sybmol"], self.exchange.value)
                    continue

                # within fungin exec, bybit returns a negative execFee when we receive amount, and positive execFee
                # when we get charged. Same goes for feeRate
                funding_fee = -Decimal(item["execFee"])
                fee_rate = -Decimal(item["feeRate"])

                items.append(
                    {
                        "id": item["execId"],
                        "exchange": ExchangeEnum.BYBIT,
                        "ticker_exchange": item["symbol"],
                        "ticker_name": ticker_name,
                        "funding_rate": fee_rate,
                        "funding_fee": funding_fee,  # + number means we receive funding, - number means we pay funding
                        "created_at": created_at,
                        "updated_at": created_at,
                    }
                )
            next_page = data["result"]["nextPageCursor"] or None

        elif self.exchange == ExchangeEnum.WOO:
            params = {}
            if symbol:
                params["symbol"] = symbol
            if from_dt:
                params["start_t"] = int(from_dt.timestamp() * 1000)
            if next_page:
                params["page"] = next_page
            if page_size:
                params["size"] = page_size
            data = await self.client.v1PrivateGetFundingFeeHistory(params)
            results = data["rows"]
            for item in results:
                if item["payment_type"] == "Pay":
                    funding_fee = -Decimal(item["funding_fee"])
                else:
                    funding_fee = Decimal(item["funding_fee"])

                # get our unified ticker name
                ticker_name = None
                markets = self.client.markets_by_id[item["symbol"]]
                for market in markets:
                    if market["type"] == "swap":
                        ticker_name = market["symbol"]
                        break

                if not ticker_name:
                    log.warning("No market found for %s on exchange %s", item["sybmol"], self.exchange.value)
                    continue

                items.append(
                    {
                        "id": item["id"],
                        "exchange": ExchangeEnum.WOO,
                        "ticker_exchange": item["symbol"],
                        "ticker_name": ticker_name,
                        "funding_rate": Decimal(item["funding_rate"]),
                        "funding_fee": funding_fee,  # + number means we receive funding, - number means we pay funding
                        "created_at": datetime.utcfromtimestamp(float(item["created_time"])),
                        "updated_at": datetime.utcfromtimestamp(float(item["updated_time"])),
                    }
                )

            next_page = (
                int(data["meta"]["current_page"]) + 1
                if len(data["rows"]) >= int(data["meta"]["records_per_page"])
                else None
            )

        elif self.exchange == ExchangeEnum.HYPERLIQUID:
            params = {"type": "userFunding", "user": self.identifier}
            if symbol:
                params.update({"symbol": symbol})
            if from_dt:
                params["startTime"] = int(from_dt.timestamp() * 1000)

            results = await self.client.request("info", method="POST", params=params)
            for item in results:
                if item["delta"]["type"] != "funding":
                    continue

                items.append(
                    {
                        "id": item["hash"],
                        "exchange": ExchangeEnum.HYPERLIQUID,
                        "ticker_exchange": item["delta"]["coin"],
                        "ticker_name": self.client.coin_to_market_id(item["delta"]["coin"]),
                        "funding_rate": Decimal(item["delta"]["fundingRate"]),
                        "funding_fee": Decimal(
                            item["delta"]["usdc"]
                        ),  # + number means we receive funding, - number means we pay funding
                        "created_at": datetime.utcfromtimestamp(float(item["time"]) / 1000),
                        "updated_at": datetime.utcfromtimestamp(float(item["time"]) / 1000),
                    }
                )

            # NOTE: we have no info about max page size from HL docs
            next_page = None

        elif self.exchange == ExchangeEnum.BINANCE:
            params = {}
            results = await self.client.fetch_funding_history(symbol, from_dt, page_size, params=params)
            for item in results:
                timestamp = item["timestamp"]
                rate_data = await self.client.fetch_funding_rate_history(
                    symbol, since=timestamp, limit=1, params={"until": timestamp}
                )
                if len(rate_data) > 1:
                    log.warning(
                        "Got %s results when fetcing funding fees for % at timestamp %s",
                        len(rate_data),
                        symbol,
                        timestamp,
                    )
                items.append(
                    {
                        "id": item["id"],
                        "exchange": ExchangeEnum.BINANCE,
                        "ticker_exchange": item["info"]["symbol"],
                        "ticker_name": item["symbol"],
                        "funding_rate": Decimal(rate_data[0]["fundingRate"]),
                        "funding_fee": Decimal(item["amount"]),
                        "created_at": datetime.utcfromtimestamp(float(timestamp) / 1000),
                        "updated_at": datetime.utcfromtimestamp(float(timestamp) / 1000),
                    }
                )
            # there is no pagination on binance, we have to implement our own
            next_page = None
        else:
            raise ECInvalidValue(f"Unknown exchange {self.exchange} provided")

        result = {
            "items": items,
            "next": next_page,
        }
        return result

    async def get_funding_fees(
        self, symbol: typing.Optional[str] = None, from_dt: typing.Optional[datetime] = None, page_size=25
    ):
        """
        Fetch funding fees
        """
        results = []
        next_page = None
        while True:
            data = await self.get_funding_fees_history(
                symbol=symbol, from_dt=from_dt, page_size=page_size, next_page=next_page
            )
            results += data["items"]
            if not data["next"]:
                break
            next_page = data["next"]

        return results

    def handle_balance_response(self, response_data):
        """ikiLeaks to destroy any remaining classified documents and informa
        Handle the response from the exchange and return the balance
        """
        balance = {
            "exchange": self.exchange.value,
            "coins": {},
        }

        if self.exchange == ExchangeEnum.BYBIT:
            # https://www.bybit.com/en/help-center/article/Maintenance-Margin-USDT-ContractUSDT_Perpetual_UTA
            # https://www.bybit.com/en/help-center/article/Maintenance-Margin-Calculation-USDC-Contract
            account_data = response_data["account_data"]
            response = response_data["balance"]
            results = response["info"]["result"]["list"]
            if len(results) > 1:
                raise ECUnexpectedError("More than one result found when fetching balances. Expected only one.")

            # Note from Bybit:
            # - If the Initial Margin Rate is equal to or greater than 100%, it means that all margin balance has been
            # deployed for your active orders and open positions and you would no longer be able to place active orders
            # that may increase your position size.
            # - Liquidation is triggered when account maintenance margin rate reaches 100%. At 80% it starts the
            # liquidation of Spot assets

            total_notional = Decimal(0)
            for position in response_data["positions"]:
                total_notional += Decimal(position["notional"])

            total_notional = total_notional.quantize(Decimal("0.00000001"))

            item = results[0]

            # Bybit has its own MMR rate stored under: item["accountMMRate"] which provides a ratio of the total
            # maintenance margin to the available margin balance in the account. This gives an overall view of how
            # much of the available margin is used to maintain positions.
            # We use "our own" calculation to get the MMR as it is more accurate and it reflects the maintenance margin
            # requirement as a percentage of the total notional value of positions. It ends up representing the
            # MMR number the same way as WOO does meaning that the same formula applies to see if we're in danger
            # of liquidation.
            total_maintenance_margin = Decimal(item["totalMaintenanceMargin"])
            mmr = (
                (total_maintenance_margin / total_notional).quantize(Decimal("0.00000001")) if total_notional else None
            )

            margin_ratio = (
                (Decimal(item["totalMarginBalance"]) / total_notional).quantize(Decimal("0.00000001"))
                if total_notional
                else None
            )

            balance.update(
                {
                    "account_type": item["accountType"],
                    "position_mode": account_data["marginMode"],  # ISOLATED_MARGIN, REGULAR_MARGIN, PORTFOLIO_MARGIN
                    "total_balance": Decimal(item["totalEquity"]),
                    "margin_balance": Decimal(item["totalMarginBalance"]),
                    "free_margin": Decimal(item["totalAvailableBalance"]),
                    "initial_margin": Decimal(item["totalInitialMargin"]),
                    "total_notional": total_notional,
                    "margin_ratio": margin_ratio,
                    "maintenance_margin": total_maintenance_margin,
                    "maintenance_margin_ratio": mmr,
                }
            )

            coin_balances = item["coin"]
            for coin in coin_balances:
                token = coin["coin"]
                balance["coins"][token] = {
                    "coin": token,
                    "usdValue": Decimal(coin["usdValue"]).quantize(Decimal("0.00000001")),
                    "equity": Decimal(coin["equity"]).quantize(Decimal("0.00000001")),
                    "balance": Decimal(coin["walletBalance"]).quantize(Decimal("0.00000001")),
                    "is_collateral": coin["collateralSwitch"],
                    "total": Decimal(str(response["total"].get(token, 0) or 0)).quantize(Decimal("0.00000001")),
                    "free": Decimal(str(response["free"].get(token, 0) or 0)).quantize(Decimal("0.00000001")),
                    # "availableToWithdraw": Decimal(coin["availableToWithdraw"]),
                    # "unrealisedPnl": Decimal(coin["unrealisedPnl"]),
                    # "cumRealisedPnl": Decimal(coin["cumRealisedPnl"]),
                    # "totalPositionIM": Decimal(coin["totalPositionIM"]),
                    # "totalOpenOrderIM": Decimal(coin["totalOpenOrderIM"]),
                    # "totalOrderIM": Decimal(coin["totalOrderIM"]),
                }

        elif self.exchange == ExchangeEnum.WOO:
            # https://support.woo.org/hc/en-001/articles/14454018487833--Margin-Margin-Ratio-margin-trading
            # https://support.woo.org/hc/en-001/articles/4718459702169--Margin-and-Leverage
            # https://support.woo.org/hc/en-001/articles/14454342431385--Liquidation-margin-trading
            # https://docs.woo.org/?python#get-account-information
            account_data = response_data["account_data"]
            response = response_data["balance"]

            total_notional = Decimal(0)
            for position in response_data["positions"]:
                total_notional += Decimal(position["notional"])

            total_notional = total_notional.quantize(Decimal("0.00000001"))

            # Note from WOO:
            # - When your margin ratio hits MMR (maintenance margin ratio), the liquidation engine will take over your
            # account and start liquidation process. Note that the MMR will rise when your position exceeds a
            # threshold.

            maintenance_margin = (total_notional * Decimal(account_data["maintenanceMarginRatio"])).quantize(
                Decimal("0.00000001")
            )
            margin_ratio = (
                (Decimal(account_data["totalCollateral"]) / total_notional).quantize(Decimal("0.00000001"))
                if total_notional
                else None
            )
            maintenance_margin_ratio = (
                (maintenance_margin / total_notional).quantize(Decimal("0.00000001")) if total_notional else None
            )

            balance.update(
                {
                    "account_type": account_data["accountMode"],
                    "position_mode": account_data["positionMode"],  # ONE_WAY or HEDGE
                    "total_balance": Decimal(account_data["totalAccountValue"]),
                    "margin_balance": Decimal(account_data["totalCollateral"]),
                    "free_margin": Decimal(account_data["freeCollateral"]),
                    "initial_margin": total_notional * Decimal(account_data["initialMarginRatio"]),
                    "total_notional": total_notional,
                    "margin_ratio": margin_ratio,  # unified with bybit (account MR) as WOO returns a general MR under marginRatio
                    "maintenance_margin": maintenance_margin,
                    "maintenance_margin_ratio": maintenance_margin_ratio,  # unified with bybit (account MMR) as WOO returns a general MMR under maintenanceMarginRatio
                }
            )

            results = response["info"]["holding"]
            for item in results:
                current_price = item["markPrice"]
                token = item["token"]
                balance["coins"][token] = {
                    "coin": token,
                    "usdValue": (Decimal(item["holding"]) * Decimal(current_price)).quantize(Decimal("0.00000001")),
                    "equity": Decimal(item["holding"]).quantize(Decimal("0.00000001")),
                    "balance": Decimal(item["availableBalance"]).quantize(Decimal("0.00000001")),
                    "total": Decimal(str(response["total"].get(token, 0) or 0)).quantize(Decimal("0.00000001")),
                    "free": Decimal(str(response["free"].get(token, 0) or 0)).quantize(Decimal("0.00000001")),
                }

        elif self.exchange == ExchangeEnum.HYPERLIQUID:
            account_data = {}  # does not exist on HL :(
            response_spot = response_data["balance_spot"]
            response = response_data["balance"]
            positions = response_data["positions"]
            total_balance = Decimal(response["total"]["USDC"]).quantize(Decimal("0.00000001"))
            margin_balance = Decimal(response["used"]["USDC"]).quantize(Decimal("0.00000001"))
            free_margin = Decimal(response["free"]["USDC"]).quantize(Decimal("0.00000001"))
            initial_margin = Decimal(sum(Decimal(position["initialMargin"]) for position in positions)).quantize(
                Decimal("0.00000001")
            )
            total_notional = Decimal(sum(Decimal(position["notional"]) for position in positions)).quantize(
                Decimal("0.00000001")
            )
            maintenance_margin = Decimal(response["info"]["crossMaintenanceMarginUsed"])
            maintenance_margin_ratio = maintenance_margin / total_balance if maintenance_margin else Decimal(0)
            margin_ratio = (margin_balance / total_notional if total_notional else Decimal(0)).quantize(
                Decimal("0.00000001")
            )

            balance.update(
                {
                    "account_type": None,  # account_data["accountMode"],
                    "position_mode": "ONE_WAY",  # only ONE_WAY is possible on HL
                    "total_balance": total_balance,
                    "margin_balance": margin_balance,
                    "free_margin": free_margin,
                    "initial_margin": initial_margin,
                    "total_notional": total_notional,
                    "margin_ratio": margin_ratio,  # unified with bybit (account MR) as WOO returns a general MR under marginRatio
                    "maintenance_margin": maintenance_margin,
                    "maintenance_margin_ratio": maintenance_margin_ratio,  # unified with bybit (account MMR) as WOO returns a general MMR under maintenanceMarginRatio
                }
            )

            # we only get USDC from type=futures request...
            token = "USDC"  # hardcoded value as they only have USDC
            balance["coins"][token] = {
                "coin": token,
                "usdValue": Decimal(response["total"][token]).quantize(Decimal("0.00000001")),
                "equity": Decimal(response["total"][token]).quantize(Decimal("0.00000001")),
                "balance": Decimal(response["free"][token]).quantize(Decimal("0.00000001")),
                # we have this to make it compatible with other dicts
                "total": Decimal(str(response["total"].get(token, 0) or 0)).quantize(Decimal("0.00000001")),
                "free": Decimal(str(response["free"].get(token, 0) or 0)).quantize(Decimal("0.00000001")),
            }

            # we get other token balances from type=spot request, we don't really care much about USDC value ... but hmm, maybe we should
            # so we know when a farm cant be open because theres not enough balance on SPOT account ... note that default account is
            # always futures on HL
            for item in response_spot["info"]["balances"]:
                token = item["coin"]
                if token == "USDC":
                    continue  # skip it ..
                balance["coins"][token] = {
                    "coin": token,
                    "usdValue": Decimal(response_spot["total"][token]).quantize(Decimal("0.00000001")),
                    "equity": Decimal(response_spot["total"][token]).quantize(Decimal("0.00000001")),
                    "balance": Decimal(response_spot["free"][token]).quantize(Decimal("0.00000001")),
                    # we have this to make it compatible with other dicts
                    "total": Decimal(str(response_spot["total"].get(token, 0) or 0)).quantize(Decimal("0.00000001")),
                    "free": Decimal(str(response_spot["free"].get(token, 0) or 0)).quantize(Decimal("0.00000001")),
                }

        elif self.exchange == ExchangeEnum.BINANCE:
            account_data = response_data["account_data"] = {}
            positions = response_data["positions"]

            balance_future = response_data["balance_future"]

            total_notional = Decimal(sum(Decimal(position["notional"]) for position in positions)).quantize(
                Decimal("0.00000001")
            )
            free_margin = Decimal(balance_future["info"]["availableBalance"])
            total_balance = Decimal(balance_future["info"]["totalWalletBalance"])
            margin_balance = Decimal(balance_future["info"]["totalMarginBalance"])
            initial_margin = Decimal(balance_future["info"]["totalPositionInitialMargin"])
            maintenance_margin = Decimal(balance_future["info"]["totalMaintMargin"])
            maintenance_margin_ratio = maintenance_margin / total_balance if maintenance_margin else Decimal(0)
            margin_ratio = (margin_balance / total_notional if total_notional else Decimal(0)).quantize(
                Decimal("0.00000001")
            )

            balance.update(
                {
                    "account_type": None,  # account_data["accountMode"],
                    "position_mode": "ONE_WAY",  # only ONE_WAY is possible on Binance??
                    "total_balance": total_balance,
                    "margin_balance": margin_balance,
                    "free_margin": free_margin,
                    "initial_margin": initial_margin,
                    "total_notional": total_notional,
                    "margin_ratio": margin_ratio,  # unified with bybit (account MR) as WOO returns a general MR under marginRatio
                    "maintenance_margin": balance_future["info"]["totalMaintMargin"],
                    "maintenance_margin_ratio": maintenance_margin_ratio,  # unified with bybit (account MMR) as WOO returns a general MMR under maintenanceMarginRatio
                }
            )

            # spot balances
            for token, val in response_data["balance"].items():
                if not isinstance(val, dict):
                    continue
                if "free" not in val and "total" not in val:
                    continue

                total = Decimal(str(val["total"] or 0)).quantize(Decimal("0.00000001"))
                free = Decimal(str(val["free"] or 0)).quantize(Decimal("0.00000001"))

                if total <= Decimal(0) or free <= Decimal(0):
                    continue

                balance["coins"][token] = {
                    "coin": token,
                    "usdValue": None,
                    "equity": None,
                    "balance": None,
                    # we have this to make it compatible with other dicts
                    "total": total,
                    "free": free,
                }

        else:
            raise ECInvalidValue(f"Unknown exchange {self.exchange} provided")

        return balance

    async def get_apikey_info(self):
        if self.exchange == ExchangeEnum.BYBIT:
            return await self.client.privateGetV5UserQueryApi()

        raise ECInvalidValue(f"{self.exchange} not supported")

    async def get_balancev2(self):
        if self.exchange == ExchangeEnum.HYPERLIQUID:
            spot_balance = await self.client.fetch_balance(params={"type": "spot"})
            futures_balance = await self.client.fetch_balance(params={"type": "futures"})
        elif self.exchange == ExchangeEnum.BYBIT:
            spot_balance = await self.client.fetch_balance()
            futures_balance = spot_balance
        elif self.exchange == ExchangeEnum.WOO:
            spot_balance = await self.client.fetch_balance()
            futures_balance = spot_balance
        elif self.exchange == ExchangeEnum.BINANCE:
            spot_balance = await self.client.fetch_balance(params={"type": "spot"})
            futures_balance = await self.client.fetch_balance(params={"type": "future"})
        else:
            raise ECInvalidValue(f"{self.exchange} not supported")

        return {
            "spot": {
                "free": spot_balance["free"],
                "used": spot_balance["used"],
                "total": spot_balance["total"],
            },
            "futures": {
                "free": futures_balance["free"],
                "used": futures_balance["used"],
                "total": futures_balance["total"],
            },
        }

    async def get_positionsv2(self, symbols, params=None):
        params = params if params else {}
        positions = await self.client.fetch_positions(symbols, params=params)
        positions = [pos for pos in positions if pos.get("notional") not in (None, 0)]
        return positions

    async def get_open_orders(self, symbol):
        resp = await self.client.fetch_open_orders(symbol)
        return resp

    async def cancel_all_orders(self, symbol):
        if not self.client.has.get("cancelAllOrders"):
            orders = await self.client.fetch_open_orders(symbol)
            output = []
            for o in orders:
                output.append(await self.cancel_order(o["id"], symbol))
                return output
        else:
            return await self.client.cancel_all_orders(symbol)

    async def close_position(self, symbol):
        if not self.client.has.get("closePosition"):
            pos = await self.client.fetch_position(symbol)
            if not pos:
                return
            qty = abs(pos["contracts"])
            price = pos["notional"] / qty
            side = "buy" if pos["side"] == "short" else "sell"
            return await self.client.create_market_order(symbol, side, qty, price, params={"reduceOnly": True})
        else:
            return await self.client.close_position(symbol)

    async def market_sell_spot(self, symbol, qty):
        try:
            qty_precision = float(self.client.amount_to_precision(symbol, qty))
            if qty_precision < qty:
                qty = qty_precision
        except ccxt_errors.InvalidOrder:
            qty = None

        if not qty:
            return

        price = None
        if self.exchange in [ExchangeEnum.HYPERLIQUID]:
            # price is required for some exchanges in order to calc. slippage
            orderbook = await self.get_orderbook(symbol)
            price = orderbook["bids"][0][0]
        live_order = await self.client.create_market_order(symbol, "sell", qty, price)
        if not live_order.get("price"):
            live_order["price"] = price
        if not live_order.get("amount"):
            live_order["amount"] = qty
        if not live_order.get("side"):
            live_order["side"] = "sell"
        return live_order

    async def cancel_order(self, oid, symbol):
        try:
            return await self.client.cancel_order(oid, symbol)
        except ccxt_errors.OrderNotFound:
            return
        except ccxt_errors.BadRequest as e:
            if "order can not be found" in e.args[0].lower():  # woo error
                return
            elif "order has been terminated" in e.args[0].lower():  # WOO error ...
                return
            raise e

    def round_qty(self, symbol: str, qty):
        try:
            proposed_qty = float(qty)
            qty = float(self.client.amount_to_precision(symbol, proposed_qty))
            # amount_to_precision does not round numbers down but it rounds it up, causing issue when selling SPOT
            # because we want to sell more than we have...
            # TIL: float(0.9) > Decimal("0.9") = True ... this is because float(0.9)  # ≈ 0.900000000000000022
            if Decimal(str(abs(qty))) > abs(proposed_qty):
                amount_precision = Decimal(str(self.client.markets[symbol]["precision"]["amount"]))
                if amount_precision >= 1:
                    # we need this for when we get "1.0" returned (eg for PURR/USDC), and if there's a trailing 0 in
                    # the quantize will assume we want to have up to 1 decimal places...
                    amount_precision = int(amount_precision)
                if self.client.precisionMode == DECIMAL_PLACES:
                    amount_precision = Decimal("10") ** -amount_precision
                qty = Decimal(str(proposed_qty)).quantize(amount_precision, rounding=ROUND_DOWN)
        except ccxt_errors.InvalidOrder:
            qty = 0.0

        return qty

    def round_price(self, symbol, price):
        try:
            return self.client.price_to_precision(symbol, float(price))
        except ccxt_errors.InvalidOrder:
            return 0
