import asyncio
from datetime import datetime, timezone, timedelta
from typing import List
from shori.utils import handle_execution_loop_errors
from shori.logger import get_logger
from ccxt.base.types import OrderRequest
from ccxt.base import errors as ccxt_errors
from shori.clearing import Order

log = get_logger(__name__)


class OMS:
    def __init__(self, client, clearing) -> None:
        self.clearing = clearing
        self.orders = {}
        self.exchange = client
        self.symbol = None
        self.lock = asyncio.Lock()
        self.since = (datetime.now(timezone.utc) - timedelta(days=3)).timestamp() * 1000
        self.startup_orders_fetched = False

    @handle_execution_loop_errors
    async def listen(self, symbol: str):
        self.symbol = symbol

        if self.startup_orders_fetched is False:
            orders = await self.exchange.fetch_open_orders(symbol)
            for o in orders:
                await self.cancel_order(o["id"], symbol)
            self.startup_orders_fetched = True
            return

        # https://docs.ccxt.com/#/ccxt.pro.manual?id=streaming-specifics
        orders = await self.exchange.watch_orders(symbol)

        for o in orders:
            oid = str(o["id"])
            status = o["status"].lower()
            log.debug("[nonblocking] Received order %s with status=%s", oid, status)

            order = {
                "id": oid,
                "status": status,
                "amount": o["amount"],
                "filled": o["filled"],
                "remaining": o["remaining"],
                "side": o["side"],
                "symbol": o["symbol"],
                "price": o["price"],
            }
            await self.clearing.update_order(o["id"], order)

    async def unsubscribe(self):
        if self.symbol:
            if self.exchange.has.get("unWatchOrders"):
                await self.exchange.un_watch_orders(self.symbol)

    async def create_orders(self, orders: List[OrderRequest], params={}):
        if not hasattr(self.exchange, "create_orders_ws"):
            raise NotImplementedError("Batch create order not supported")

        orders = await self.exchange.create_orders_ws(orders, params)
        for order in orders:
            if order["id"]:
                log.debug(
                    "Created %s order %s with status=%s",
                    order["symbol"],
                    order["id"],
                    order["status"],
                )
                o = Order(
                    order_id=str(order["id"]),
                    symbol=order["symbol"],
                    side=order["side"],
                    price=order["price"],
                    amount=order["amount"],
                    status="open",
                )
                log.debug(o)
                await self.clearing.process_new_order(o)
            else:
                log.error("Error creating order for symbol %s: %s", order["symbol"], order)
        return orders

    async def create_order(self, symbol, price, amount, side, reduce_only=False):
        """
        Create a new limit order on the exchange.

        Returns:
            dict: Order details if successful, containing:
                - id: Order ID
                - amount: Order quantity
                - price: Order price
                - symbol: Trading pair
                - side: Order side
            None: If order creation fails

        Note:
            - Uses WebSocket API if available (createOrderWs)
            - All orders are created as post-only limit orders
            - Func does not use self.lock, and you should use it outside of this function to avoid race-conditions
        """
        if self.exchange.name.lower() == "kraken":
            params = {"oflags": "post"}
        else:
            params = {"timeInForce": "PO"}
        if reduce_only:
            params.update({"reduceOnly": reduce_only})

        if self.exchange.name.lower() == "hyperliquid":
            del params["timeInForce"]

        try:
            func = self.exchange.create_order
            if bool(self.exchange.has.get("createOrderWs")):
                func = self.exchange.create_order_ws

            order = await func(symbol, "limit", side, amount, price, params=params)
        except Exception:
            log.exception("Error creating order")
            # TODO: raise? it could be: ccxt.base.errors.InsufficientFunds: bybit {"reqId":"2404","retCode":170131,"retMsg":"Insufficient balance.","op":"order.create","data":{},"header":{"X-Bapi-Limit":"
            raise

        order.update(
            {
                "amount": amount,
                "price": price,
                "symbol": symbol,
                "side": side,
            }
        )

        if order["id"]:
            log.debug(
                "Created %s order %s with status=%s",
                symbol,
                order["id"],
                order["status"],
            )
            o = Order(
                order_id=str(order["id"]),
                symbol=order["symbol"],
                side=order["side"],
                price=order["price"],
                amount=order["amount"],
                status="open",
            )
            log.debug(o)
            await self.clearing.process_new_order(o)
            return order
        else:
            log.error("Error creating order for symbol %s: %s", symbol, order)

    async def edit_order(
        self,
        oid: str,
        symbol: str,
        new_price=None,
        new_amount=None,
        new_side=None,
        reduce_only=False,
    ):
        """
        Edit an existing order on the exchange.

        Returns:
            dict: Updated order details if successful, containing:
                - id: Order ID
                - amount: Updated quantity
                - price: Updated price
                - symbol: Trading pair
                - side: Order side
            None: If order edit fails or order doesn't exist

        Note:
            - Uses WebSocket API if available (editOrderWs)
            - All orders are created as post-only limit orders
            - Order will be removed from local cache if it's found to be non-existent on exchange
            - Func does not use self.lock, and you should use it outside of this function to avoid race-conditions
        """
        old_order = self.clearing.orders.get(oid)
        if not old_order:
            return
        if self.exchange.name.lower() == "kraken":
            params = {"oflags": "post"}
        else:
            params = {"timeInForce": "PO"}
        if reduce_only:
            params.update({"reduceOnly": reduce_only})

        if self.exchange.name.lower() == "hyperliquid":
            del params["timeInForce"]

        new_amount = new_amount if new_amount else old_order.amount
        new_price = new_price if new_price else old_order.price
        new_side = new_side if new_side else old_order.side

        try:
            func = self.exchange.edit_order
            if bool(self.exchange.has.get("editOrderWs")):
                func = self.exchange.edit_order_ws
            order = await func(oid, symbol, "limit", new_side, new_amount, new_price, params=params)
            log.info("Edited order %s", order)
        except ccxt_errors.BadRequest as e:
            log.info(e)
            if "order can not be found" in e.args[0].lower():  # woo error
                return
            elif "order has been terminated" in e.args[0].lower():  # WOO error ...
                return
            elif "can't edit this open order because it doesn't exist anymore" in e.args[0].lower():  # woo error
                return
            elif "the order didn't update" in e.args[0].lower():
                # we dont remove the order because the resp does not indicate that order does not exist, but that it
                # didn't get updated
                return
            elif (
                "order remains unchanged as the parameters entered match the existing ones" in e.args[0].lower()
            ):  # bbt error
                return
            else:
                raise e
        except Exception as e:
            log.exception(
                "Exception when editing an order %s for symbol %s",
                oid,
                symbol,
                exc_info=e,
            )
            await asyncio.sleep(0)
        else:
            order.update(
                {
                    "amount": new_amount,
                    "price": new_price,
                    "symbol": symbol,
                    "side": new_side,
                }
            )
            if order["id"]:
                return order

    async def cancel_order(self, oid, symbol):
        try:
            func = self.exchange.cancel_order
            if bool(self.exchange.has.get("cancelOrderWs")):
                func = self.exchange.cancel_order_ws

            await func(oid, symbol)
        except ccxt_errors.OrderNotFound:
            return
        except ccxt_errors.BadRequest as e:
            if "order can not be found" in e.args[0].lower():  # woo error
                return
            elif "order has been terminated" in e.args[0].lower():  # WOO error ...
                return
            elif "can't edit this open order because it doesn't exist anymore" in e.args[0].lower():  # woo error
                return
            elif "the order didn't update" in e.args[0].lower():
                # we dont remove the order because the resp does not indicate that order does not exist, but that it
                # didn't get updated
                return
            else:
                raise e
        except Exception as e:
            log.exception(
                "Exception when cancelling order %s for symbol %s",
                oid,
                symbol,
                exc_info=e,
            )

    async def cancel_all_orders(self):
        async with self.lock:
            orders = await self.get_orders()
            symbols = [o.symbol for o in orders]
            log.info("Cancelling %s orders.", len(symbols))

            # NOTE: we cancel step-by-step because our `cancel_order` method already has logic implemented to
            # handle errors and cleanup orders form local memory
            output = []
            for o in orders:
                output.append(await self.cancel_order(o.order_id, o.symbol))
            return output

    async def get_orders(self) -> List[Order]:
        orders = []
        cached_orders = list(self.clearing.orders.values())
        for order in cached_orders:
            if order.status not in ["open"]:
                continue
            orders.append(order)
        return orders
