import asyncio

from shori.utils import handle_execution_loop_errors
from shori.clearing import Trade
from datetime import datetime, timezone, timedelta


class Trades:
    """Tracks position size and average entry price."""

    def __init__(self, client, oms, clearing, since=None, symbol=None):
        self.clearing = clearing
        self.exchange = client
        self.write_lock = asyncio.Lock()
        self.oms = oms
        self.symbol = symbol

    @handle_execution_loop_errors
    async def listen(self):
        # https://docs.ccxt.com/#/ccxt.pro.manual?id=streaming-specifics
        if not self.exchange.has.get("watchMyTrades"):
            raise Exception(f"{self.exchange.name} does not have support for watchMyTrades")

        trades = await self.exchange.watch_my_trades(
            self.symbol,
            since=int((datetime.now(timezone.utc) - timedelta(seconds=10)).timestamp() * 1000),
        )

        for t in trades:
            oid = t["order"]
            if not oid:
                oid = t["info"]["oid"]  # hyperliquid ccxt bug ...

            trade = Trade(
                trade_id=str(t["id"]),
                order_id=str(oid),
                symbol=t["symbol"],
                side=t["side"],
                price=t["price"],
                amount=t["amount"],
                timestamp=t["timestamp"],
            )
            await self.clearing.process_trade(trade)

    async def unsubscribe(self):
        if self.symbol:
            if self.exchange.has.get("unWatchMyTrades"):
                await self.exchange.un_watch_my_trades(self.symbol)
